import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Bike, Mountain, Car, Baby, Zap, MessageSquare } from 'lucide-react'
import { motion } from 'framer-motion'
import './RentBike.css'

const RentBike = () => {
  const navigate = useNavigate()
  const [selectedType, setSelectedType] = useState('')
  const [selectedReturnDate, setSelectedReturnDate] = useState('')
  const [review, setReview] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const bikeTypes = [
    { 
      id: 'mountain', 
      name: 'Горный', 
      icon: <Mountain size={32} strokeWidth={2} />, 
      description: 'Для горных трасс и бездорожья',
      color: '#22C55E'
    },
    { 
      id: 'city', 
      name: 'Городской', 
      icon: <Car size={32} strokeWidth={2} />, 
      description: 'Комфортные поездки по городу',
      color: '#16A34A'
    },
    { 
      id: 'kids', 
      name: 'Детский', 
      icon: <Baby size={32} strokeWidth={2} />, 
      description: 'Безопасные велосипеды для детей',
      color: '#4ADE80'
    },
    { 
      id: 'electric', 
      name: 'Электровелосипед', 
      icon: <Zap size={32} strokeWidth={2} />, 
      description: 'С электромотором для дальних поездок',
      color: '#22C55E'
    }
  ]

  const getPrice = (type) => {
    const prices = {
      'mountain': 1500,
      'city': 1200,
      'kids': 800,
      'electric': 2000
    }
    return prices[type] || 0
  }

  const getModel = (type) => {
    const models = {
      'mountain': 'Trek Mountain Pro',
      'city': 'City Cruiser 2024',
      'kids': 'Kids Bike Junior',
      'electric': 'Electric Speed X1'
    }
    return models[type] || ''
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedType && selectedReturnDate) {
      const savedRentals = localStorage.getItem('bikeRentals')
      const rentals = savedRentals ? JSON.parse(savedRentals) : { active: [], past: [] }
      
      const newRental = {
        id: Date.now(),
        model: getModel(selectedType),
        type: bikeTypes.find(t => t.id === selectedType).name,
        date: new Date().toISOString().split('T')[0],
        returnDate: selectedReturnDate,
        price: getPrice(selectedType),
        status: 'active',
        review: review || null
      }

      rentals.active.push(newRental)
      localStorage.setItem('bikeRentals', JSON.stringify(rentals))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedType('')
        setSelectedReturnDate('')
        setReview('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="rent-bike">
      <section className="page-intro">
        <motion.div
          className="intro-content"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="intro-icon-wrapper">
            <Bike className="intro-main-icon" size={64} color="#22C55E" />
          </div>
          <h1 className="intro-title">Арендовать велосипед</h1>
          <p className="intro-description">
            Выберите тип велосипеда и дату возврата для комфортной поездки
          </p>
        </motion.div>
      </section>

      <div className="container">
        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.form 
            className="booking-form" 
            onSubmit={handleSubmit}
            initial={{ scale: 0.95 }}
            animate={{ scale: 1 }}
            transition={{ type: "spring", stiffness: 100, damping: 10, delay: 0.3 }}
          >
            <div className="form-group">
              <label className="form-label">
                <Bike className="label-icon" size={20} strokeWidth={2} />
                Выберите тип велосипеда
              </label>
              <div className="types-row">
                {bikeTypes.map(type => (
                  <motion.button
                    key={type.id}
                    type="button"
                    className={`type-button ${selectedType === type.id ? 'selected' : ''}`}
                    onClick={() => setSelectedType(type.id)}
                    whileHover={{ scale: 1.05 }}
                    whileTap={{ scale: 0.95 }}
                    style={{ borderColor: selectedType === type.id ? 'var(--primary)' : 'rgba(34, 197, 94, 0.2)' }}
                  >
                    <div className="type-button-icon" style={{ background: selectedType === type.id ? 'rgba(34, 197, 94, 0.15)' : 'transparent' }}>
                      {type.icon}
                    </div>
                    <div className="type-button-content">
                      <h3 className="type-button-name">{type.name}</h3>
                      <p className="type-button-price">{getPrice(type.id)} ₽/день</p>
                    </div>
                  </motion.button>
                ))}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="returnDate" className="form-label">
                <Calendar className="label-icon" size={20} strokeWidth={2} />
                Дата возврата
              </label>
              <input
                type="date"
                id="returnDate"
                className="form-input"
                value={selectedReturnDate}
                onChange={(e) => setSelectedReturnDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <div className="form-group">
              <label htmlFor="review" className="form-label">
                <MessageSquare className="label-icon" size={20} strokeWidth={2} />
                Отзыв (необязательно)
              </label>
              <textarea
                id="review"
                className="form-textarea"
                value={review}
                onChange={(e) => setReview(e.target.value)}
                rows={4}
                placeholder="Оставьте отзыв после аренды..."
              />
            </div>

            {selectedType && selectedReturnDate && (
              <div className="price-summary">
                <div className="summary-row">
                  <span>Стоимость аренды:</span>
                  <span className="summary-price">{getPrice(selectedType)} ₽</span>
                </div>
                <div className="summary-row">
                  <span>Количество дней:</span>
                  <span>{Math.ceil((new Date(selectedReturnDate) - new Date()) / (1000 * 60 * 60 * 24))} дней</span>
                </div>
                <div className="summary-total">
                  <span>Итого:</span>
                  <span className="total-price">
                    {getPrice(selectedType) * Math.ceil((new Date(selectedReturnDate) - new Date()) / (1000 * 60 * 60 * 24))} ₽
                  </span>
                </div>
              </div>
            )}

            <motion.button 
              type="submit" 
              className="submit-btn" 
              disabled={!selectedType || !selectedReturnDate}
              whileHover={{ translateY: -3, boxShadow: '0 10px 30px rgba(34, 197, 94, 0.45)' }}
              whileTap={{ scale: 0.98 }}
            >
              <CheckCircle size={20} strokeWidth={2} className="btn-icon" />
              Арендовать велосипед
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={24} strokeWidth={2} className="success-icon" />
                <span>Велосипед успешно арендован!</span>
              </motion.div>
            )}
          </motion.form>
        </motion.section>

        <motion.section 
          className="info-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="info-card">
            <h2 className="info-title">Почему выбирают нас?</h2>
            <ul className="info-list">
              <li>Широкий выбор велосипедов разных типов</li>
              <li>Гибкие условия аренды и доступные цены</li>
              <li>Все велосипеды в отличном техническом состоянии</li>
              <li>Удобные пункты выдачи по всему городу</li>
              <li>Круглосуточная поддержка клиентов</li>
            </ul>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default RentBike

